/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2024-02-07 12:06:34 +0000 (Wed, 07 Feb 2024) $ $Rev: 1793 $
 */
package de.consist.bmu.rule;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.error.BMUParseException;
import de.consist.bmu.rule.impl.BMUDokumentImpl;
import de.consist.bmu.rule.util.XmlUtils;

/**
 * Hilfsklasse zum Erzeugen und Serialisieren von Document-Instanzen.
 * 
 * @author jannighoefer
 */
public final class DocumentController {

    private static final String DISALLOW_DOCTYPE_DECL_FEATURE = "http://apache.org/xml/features/disallow-doctype-decl";

    private static final Log LOGGER = LogFactory
            .getLog(DocumentController.class);

    private DocumentController() {
    }

    /**
     * @param data
     *            Die Daten
     * @return Das Document
     * @throws BMUException
     *             BMUException
     */
    public static Document parse(byte[] data) throws BMUException {
        return parse(new ByteArrayInputStream(data));
    }

    /**
     * @param data
     *            Die Daten
     * @param ruleSet 
     *            Das RuleSet
     * @return Das BMUDokument
     * @throws BMUParseException
     *             BMUParseException
     */
    public static BMUDokument parseBMU(byte[] data, RuleSet ruleSet) throws BMUParseException {
        return BMUDokumentImpl.parse(new ByteArrayInputStream(data), data.length, ruleSet);
    }

    /**
     * @param is
     *            Der InputStream
     * @param size
     *            Die Groesse des Dokuments
     * @param ruleSet 
     *            Das RuleSet
     * @return Das BMUDokument
     * @throws BMUParseException
     *             BMUParseException
     */
    public static BMUDokument parseBMU(InputStream is, int size, RuleSet ruleSet) throws BMUParseException {
        return BMUDokumentImpl.parse(is, size, ruleSet);
    }

    /**
     * @param is
     *            Der InputStream
     * @param ruleSet 
     *            Das RuleSet
     * @return Das BMUDokument
     * @throws BMUParseException
     *             BMUParseException
     */
    public static BMUDokument parseBMU(InputStream is, RuleSet ruleSet) throws BMUParseException {
        return BMUDokumentImpl.parse(is, ruleSet);
    }

    /**
     * @param file
     *            Die Datei
     * @param ruleSet 
     *            Das RuleSet
     * @return Das BMUDokument
     * @throws BMUParseException
     *             BMUParseException
     */
    public static BMUDokument parseBMU(File file, RuleSet ruleSet) throws BMUParseException {
        return BMUDokumentImpl.parse(file, ruleSet);
    }

    /**
     * @param resource
     *            Der Name der Resource
     * @param ruleSet 
     *            Das RuleSet
     * @return Das BMUDokument
     * @throws BMUParseException
     *             BMUParseException
     */
    public static BMUDokument parseBMU(String resource, RuleSet ruleSet) throws BMUParseException {
        return BMUDokumentImpl.parse(DocumentController.class
                .getResourceAsStream(resource), -1, ruleSet);
    }

    /**
     * @param resource
     *            Der Name der Resource
     * @return Das Document
     * @throws BMUException
     *             BMUException
     */
    public static Document parse(String resource) throws BMUException {
        return parse(DocumentController.class.getResourceAsStream(resource));
    }

    /**
     * @param file
     *            Das File
     * @return Das Document
     * @throws BMUException
     *             BMUException
     */
    public static Document parse(File file) throws BMUException {
        try {
            return parse(new FileInputStream(file));
        } catch (FileNotFoundException e) {
            throw new BMUException("error parsing file: "
                    + file.getAbsolutePath(), e);
        }
    }

    /**
     * @param is
     *            Der InputStream
     * @return Das Document
     * @throws BMUException
     *             BMUException
     */
    public static Document parse(InputStream is) throws BMUException {
        Document doc = null;
//        String docBuilderFactory = System.getProperty("javax.xml.parsers.DocumentBuilderFactory");
//        LOGGER.debug("javax.xml.parsers.DocumentBuilderFactory: " + docBuilderFactory);
//        if (docBuilderFactory == null) {
//        	System.setProperty("javax.xml.parsers.DocumentBuilderFactory", "org.apache.xerces.jaxp.DocumentBuilderFactoryImpl");
//        }
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        LOGGER.debug("DocumentBuilderFactory: " + dbf.getClass().getName());
        dbf.setValidating(false); // TODO Auf true umsetzen!
        dbf.setNamespaceAware(true);
        ParserErrorHandler eh = new ParserErrorHandler();

        try {
            dbf.setFeature(DISALLOW_DOCTYPE_DECL_FEATURE, true);
            DocumentBuilder db = dbf.newDocumentBuilder();
            db.setErrorHandler(eh);
            doc = db.parse(is);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Document-Impl: " + doc.getClass().getName());
                LOGGER.debug("XML Dokument geladen: "
                        + doc.getFirstChild().getNodeName());
            }
            if (eh._errors > 0 || eh._fatals > 0) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("<parse> Schema validation failed, errors: "
                            + eh._errors + ", fatals: " + eh._fatals);
                }
                throw new BMUException(
                        "Die Daten enthalten kein gltiges XML-Dokument.");
            }
        } catch (ParserConfigurationException e) {
            LOGGER.warn("<parse> Fehler beim Parsen: " + e.getMessage(), e);
            throw new BMUException("<parse> Fehler beim Parsen des Documents",
                    e);
        } catch (SAXException e) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("<parse> Fehler beim Parsen: "
                        + DocumentController.exceptionToString(e), e);
                // TODO Hier die ersten 16 Zeichen des geparseden Inhalts
                // ausgeben
            }
            throw new BMUException(
                    "Die Daten enthalten kein gltiges XML-Dokument ("
                            + DocumentController.exceptionToString(e) + ")", e);
        } catch (IOException e) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("<parse> Fehler beim Parsen: " + e.getMessage(), e);
            }
            throw new BMUException("<parse> Fehler beim Parsen des Documents",
                    e);
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException e) {
                LOGGER.warn("Closing InputStream failed", e);
            }
        }
        return doc;
    }

    private static String exceptionToString(SAXException e) {
        String retVal = "";
        if (e instanceof SAXParseException) {
            SAXParseException e2 = (SAXParseException) e;
            retVal = e2.getMessage() + " (line "
                    + Integer.toString(e2.getLineNumber()) + ", column "
                    + Integer.toString(e2.getColumnNumber()) + ")";
        } else {
            retVal = e.getMessage();
        }
        return retVal;
    }

    /** Ausbund an Genialitt. */
    private static class ParserErrorHandler implements ErrorHandler {
        private static final int MAX_ERROR_LOG = 10;
        private int _errors = 0;
        private int _fatals = 0;

        /** Implementierte org.xml.sax.ErrorHandler-Methode. */
        public void warning(SAXParseException exception) throws SAXException {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("<warning>: " + exceptionToString(exception),
                        exception);
            }
        }

        /** Implementierte org.xml.sax.ErrorHandler-Methode. */
        public void error(SAXParseException exception) throws SAXException {
            _errors++; // Oder Exception werfen?
            if (_errors <= MAX_ERROR_LOG && LOGGER.isDebugEnabled()) {
                // _logger.error("<error>", exception); // Zuviel Gelaber...
                LOGGER.debug("<error>: " + exceptionToString(exception));
            }
        }

        /** Implementierte org.xml.sax.ErrorHandler-Methode. */
        public void fatalError(SAXParseException exception) throws SAXException {
            _fatals++; // Oder Exception werfen?
            if (LOGGER.isDebugEnabled()) {
                LOGGER.info("<fatalError>: " + exceptionToString(exception),
                        exception);
            } else {
                LOGGER.info("<fatalError>: " + exceptionToString(exception));
            }
        }
    }

    /**
     * @param doc
     *            Das Document
     * @return Die Daten
     * @throws BMUException
     *             BMUException
     */
    public static byte[] serialize(Document doc) throws BMUException {
        return XmlUtils.serialize(doc);
    }

    /**
     * Liefert das Referenzdatum bezglich der Auswertungen der Prfungen.
     * Typischerweise den Zeitpunkt der ersten Signatur.
     * 
     * @param doc
     *            Das BMUDokument
     * @return Das Referenzdatum
     * @deprecated 
     */
    @Deprecated
    public static Date getReferenzDatum(BMUDokument doc) {
        return doc.getMessageType().getFirstSignTime();
    }
}
