/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2014-08-27 12:06:51 +0000 (Wed, 27 Aug 2014) $ $Rev: 1274 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.impl;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.jamonapi.Monitor;
import com.jamonapi.MonitorFactory;

import de.consist.bmu.rule.BMUDokument;
import de.consist.bmu.rule.BMUMessageType;
import de.consist.bmu.rule.Rule;
import de.consist.bmu.rule.RuleDef;
import de.consist.bmu.rule.RuleResult;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.util.DateUtils;

/**
 * Abstrakte Basis-Implementierung einer Pruefregel. Alle Pruefregeln muessen
 * diese Klasse erweitern, der Name muss mit 'RuleImpl' beginnen. Der 'name' in
 * der Regeldefinition muss dann dem variablen Teil des Klassennamens
 * entsprechen, also wird z.B. die Regel 'RuleImplSchemaValidation' durch eine
 * Regeldefinition mit dem Namen 'SchemaValidation' adressiert. Alle
 * Regelimplementierungen muessen im package 'de.consist.bmu.rule.impl'
 * erfolgen.
 * 
 * @author srossbroich
 * 
 */
public abstract class RuleImpl implements Rule, Serializable {

    private static final long serialVersionUID = 1L;

    private static final Log LOGGER = LogFactory.getLog(RuleImpl.class);

    private static boolean _jaMonEnabled = false;

    private RuleDef _ruleDef;

    /** Konstante fr Zeilenumbruch im PDF. */
    protected static final String NL = "\n"; 
    // protected static final String NL = "\u00a3"; // Pound -> "<br />" 

    protected RuleImpl(RuleDef ruleDef) {
        _ruleDef = ruleDef;
    }

    final List<RuleResult> executeInternal(BMUDokument bmuDok)
            throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        if (isEnabled(bmuDok)) {
            Monitor mon = null;
            if (_jaMonEnabled) {
                mon = MonitorFactory.getTimeMonitor(getClass().getName() + "-"
                        + _ruleDef.getId() + ":"
                        + bmuDok.getMessageType().getEnumType().toString());
                mon.start();
            }
            try {
                synchronized (bmuDok.getDocument()) {
                    ruleResultList = execute(bmuDok);
                }
            } finally {
                if (mon != null) {
                    mon.stop();
                }
            }
            if (mon != null) {
                LOGGER.debug(mon);
            }
        }
        return ruleResultList;
    }

    /**
     * @return the _ruleDef
     */
    public final RuleDef getRuleDef() {
        return _ruleDef;
    }

    /**
     * {@inheritDoc}
     */
    public boolean isEnabled(BMUDokument bmuDok) {
    	if (bmuDok != null) {
	        BMUMessageType msgType = bmuDok.getMessageType();
	        if (!_ruleDef.isEnabled()) {
	            if (LOGGER.isTraceEnabled()) {
	                LOGGER.trace("Regel " + _ruleDef.getName()
	                        + " ist deaktiviert.");
	            }
	            return false;
	        } else if (_ruleDef.getDocTypes() != null
	                && !_ruleDef.getDocTypes().contains(
	                        msgType.getEnumType().toString())) {
	            if (LOGGER.isTraceEnabled()) {
	                LOGGER.trace("Regel " + _ruleDef.getName() + " ist nicht fr "
	                        + msgType.getEnumType().toString() + ".");
	            }
	            return false;
	        } else if (_ruleDef.getUseSignatureDate() && _ruleDef.getKeyDate().after(
	                bmuDok.getMessageType().getLastSignTime())) {
	            if (LOGGER.isTraceEnabled()) {
	                LOGGER.trace("Regel "
	                        + _ruleDef.getName()
	                        + " wird erst ab "
	                        + DateUtils.toDateTimeString(_ruleDef.getKeyDate())
	                        + " angewandt, das Datum der jngsten Signatur ist "
	                        + DateUtils.toDateTimeString(bmuDok.getMessageType()
	                                .getLastSignTime()) + ".");
	            }
	            return false;
	        } else if (_ruleDef.getUseSignatureDate() && _ruleDef.getKeyToDate().before(
	                bmuDok.getMessageType().getFirstSignTime())) {
	            if (LOGGER.isTraceEnabled()) {
	                LOGGER.trace("Regel "
	                        + _ruleDef.getName()
	                        + " wird erst ab "
	                        + DateUtils.toDateTimeString(_ruleDef.getKeyDate())
	                        + " angewandt, das Datum der ltesten Signatur ist "
	                        + DateUtils.toDateTimeString(bmuDok.getMessageType()
	                                .getFirstSignTime()) + ".");
	            }
	            return false;
	        } else if (!_ruleDef.getUseSignatureDate() && _ruleDef.getKeyDate().after(
	                bmuDok.getMessageType().getFirstSignTime())) {
	            if (LOGGER.isTraceEnabled()) {
	                LOGGER.trace("Regel "
	                        + _ruleDef.getName()
	                        + " wird erst ab "
	                        + DateUtils.toDateTimeString(_ruleDef.getKeyDate())
	                        + " angewandt, das Referenzdatum ist "
	                        + DateUtils.toDateTimeString(bmuDok.getMessageType()
	                                .getFirstSignTime()) + ".");
	            }
	            return false;
	        } else if (!_ruleDef.getUseSignatureDate() && _ruleDef.getKeyToDate().before(
	                bmuDok.getMessageType().getFirstSignTime())) {
	            if (LOGGER.isTraceEnabled()) {
	                LOGGER.trace("Regel "
	                        + _ruleDef.getName()
	                        + " wird nur bis "
	                        + DateUtils.toDateTimeString(_ruleDef.getKeyToDate())
	                        + " angewandt, das Referenzdatum ist "
	                        + DateUtils.toDateTimeString(bmuDok.getMessageType()
	                                .getFirstSignTime()) + ".");
	            }
	            return false;
	        }
    	} else {
    		return _ruleDef.isEnabled();
    	}
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("Regel " + _ruleDef.getName() + " wird angewandt!");
        }
        return true;
    }

    /**
     * @param ruleDef
     *            RuleDef
     * @return RuleImpl
     * @throws BMUException
     *             BMUException
     */
    public static RuleImpl newRuleImpl(RuleDef ruleDef) throws BMUException {
        RuleImpl ruleImpl = null;
        try {
            String className = ruleDef.getName();
            if (!ruleDef.getName().contains(".")) {
                className = "de.consist.bmu.rule.impl.RuleImpl" + className;
            }
            LOGGER.debug("creating instance of class '" + className + "'");
            Class<?> ruleClass = Class.forName(className);
            if (ruleClass != null && RuleImpl.class.isAssignableFrom(ruleClass)) {
                ruleImpl = (RuleImpl) ruleClass.getConstructor(RuleDef.class)
                        .newInstance(ruleDef);
            } else {
                LOGGER.error("Error instanciating RuleImpl from RuleDef: "
                        + ruleDef);
            }
        } catch (IllegalArgumentException e) {
            LOGGER.error("Error creating Rule from RuleDef: " + ruleDef, e);
            throw new BMUException("Error creating Rule from RuleDef: "
                    + ruleDef, e);
        } catch (SecurityException e) {
            LOGGER.error("Error creating Rule from RuleDef: " + ruleDef, e);
            throw new BMUException("Error creating Rule from RuleDef: "
                    + ruleDef, e);
        } catch (InstantiationException e) {
            LOGGER.error("Error creating Rule from RuleDef: " + ruleDef, e);
            throw new BMUException("Error creating Rule from RuleDef: "
                    + ruleDef, e);
        } catch (IllegalAccessException e) {
            LOGGER.error("Error creating Rule from RuleDef: " + ruleDef, e);
            throw new BMUException("Error creating Rule from RuleDef: "
                    + ruleDef, e);
        } catch (InvocationTargetException e) {
            LOGGER.error("Error creating Rule from RuleDef: " + ruleDef, e);
            throw new BMUException("Error creating Rule from RuleDef: "
                    + ruleDef, e);
        } catch (NoSuchMethodException e) {
            LOGGER.error("Error creating Rule from RuleDef: " + ruleDef, e);
            throw new BMUException("Error creating Rule from RuleDef: "
                    + ruleDef, e);
        } catch (ClassNotFoundException e) {
            LOGGER.error("Error creating Rule from RuleDef: " + ruleDef, e);
            throw new BMUException("Error creating Rule from RuleDef: "
                    + ruleDef, e);
        }
        return ruleImpl;
    }

    /**
     * @return the _jaMonEnabled
     */
    public static boolean isJaMonEnabled() {
        return _jaMonEnabled;
    }

    /**
     * @param jaMonEnabled
     *            the jaMonEnabled to set
     */
    public static void setJaMonEnabled(boolean jaMonEnabled) {
        RuleImpl._jaMonEnabled = jaMonEnabled;
    }
}
