/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2013-06-28 08:24:11 +0000 (Fri, 28 Jun 2013) $ $Rev: 1091 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.xml.xpath.XPathExpressionException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;

import de.consist.bmu.rule.BMUDokument;
import de.consist.bmu.rule.BMUMessageType;
import de.consist.bmu.rule.RuleDef;
import de.consist.bmu.rule.RuleResult;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.xpath.XPathFassade;

/**
 * @author srossbroich
 * 
 */
public final class RuleImplFixedText extends RuleImpl {

    private static final long serialVersionUID = 1L;

    private static final Log LOGGER = LogFactory.getLog(RuleImplFixedText.class);

    /**
     * XPath-Ausdruecke fuer feste Texte (fixed text) in Nachweisen (ENS).
     */
    private static final String[] XPATH_ENS_FIXED_TEXT = {
            "en:VerantwortlicheErklaerung/en:Versicherung",
            "en:Erklaerung/en:Versicherung", "en:HinweisGebuehr",
            "en:HinweisRechtsbehelf" };

    /**
     * XPath-Ausdruecke fuer feste Texte (fixed text) in der Freistellung (FR).
     */
    private static final String[] XPATH_FR_FIXED_TEXT = { "en:Antrag/en:Text",
            "en:HinweisGebuehr", "en:HinweisRechtsbehelf" };

    /**
     * XPath-Ausdruecke fuer feste Texte (fixed text) im Begleitschein (BGS).
     */
    private static final String[] XPATH_BGS_FIXED_TEXT = { "bgs:Erzeuger/bgs:Versicherung", 
        "bgs:Befoerderer/bgs:VersicherungErsterBef",
        "bgs:Befoerderer/bgs:VersicherungWeitererBef",
        "bgs:Lager/bgs:Versicherung",
        "bgs:Entsorger/bgs:Versicherung"};

    /**
     * XPath-Ausdruecke fuer feste Texte (fixed text) im ergaenzenden Formblatt
     * (EGF).
     */
    private static final String[] XPATH_EGF_FIXED_TEXT = {
            "en:Nummer1_Antrag/en:Text", "en:Nummer1_Antrag/en:TextVollmacht",
            "en:Nummer2_Bevollmaechtigung/en:Text",
            "en:Nummer3_Beauftragung/en:Text" };

    /**
     * XPath-Ausdruecke fuer feste Texte (fixed text) in Registrierungsantraegen (RGA).
     */
    private static final String[] XPATH_RGA_FIXED_TEXT = {
            "zks:Registrierungsantrag/zks:Versicherung", "zks:RegistrierungsantragZKS/zks:Versicherung" };

    /**
     * @param ruleDef
     *            RuleDef
     */
    public RuleImplFixedText(RuleDef ruleDef) {
        super(ruleDef);
    }

    private List<RuleResult> evalFixedText(Document doc, String[] xPathes)
            throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        int index = 1;
        for (String xpFT : xPathes) {
            try {
                if (XPathFassade.getInstance().evalBool(doc.getDocumentElement(),
                        "/descendant::" + xpFT + "=''")) {
                    ruleResultList.add(new RuleResultImpl(getRuleDef(), index++, xpFT));
                }
            } catch (XPathExpressionException e) {
                throw new BMUException("Fehler beim Prfen", e);
            }
        }
        return ruleResultList;
    }

    /**
     * {@inheritDoc}
     */
    public List<RuleResult> execute(BMUDokument bmuDok) throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        Document doc = bmuDok.getDocument();
        BMUMessageType msgType = bmuDok.getMessageType();
        switch (msgType.getEnumType()) {
        case ENSNDokument:
            ruleResultList.addAll(evalFixedText(doc, XPATH_ENS_FIXED_TEXT));
            break;
        case BGSDokument:
            ruleResultList.addAll(evalFixedText(doc, XPATH_BGS_FIXED_TEXT));
            break;
        case FRDokument:
            ruleResultList.addAll(evalFixedText(doc, XPATH_FR_FIXED_TEXT));
            break;
        case EGFDokument:
            ruleResultList.addAll(evalFixedText(doc, XPATH_EGF_FIXED_TEXT));
            break;
        case RegisterAnforderung:
            ruleResultList.addAll(evalFixedText(doc, XPATH_RGA_FIXED_TEXT));
            break;
        default:
            LOGGER.warn("unexpected message-type: " + msgType.getEnumType());
        }
        return ruleResultList;
    }

    /**
     * @return Die technische Dokumentation der Implementierung
     */
    public static String getTechDoc() {
        StringBuilder builder = new StringBuilder();
        builder.append("Fr die Prfung werden folgende XPath-Ausdrcke verwendet:");
        builder.append(RuleImpl.NL);
        builder.append("ENSNDokument: " + Arrays.deepToString(XPATH_ENS_FIXED_TEXT));
        builder.append(RuleImpl.NL);
        builder.append("BGSDokument: " + Arrays.deepToString(XPATH_BGS_FIXED_TEXT));
        builder.append(RuleImpl.NL);
        builder.append("FRDokument: " + Arrays.deepToString(XPATH_FR_FIXED_TEXT));
        builder.append(RuleImpl.NL);
        builder.append("EGFDokument: " + Arrays.deepToString(XPATH_EGF_FIXED_TEXT));
        builder.append(RuleImpl.NL);
        builder.append("Registeranforderung: " + Arrays.deepToString(XPATH_RGA_FIXED_TEXT));
        return builder.toString();
    }
}
