/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2024-02-07 09:53:51 +0000 (Wed, 07 Feb 2024) $ $Rev: 1790 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.schema;

import java.io.Serializable;
import java.security.NoSuchProviderException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlType;
import jakarta.xml.bind.annotation.adapters.XmlAdapter;
import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeConstants;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.xpath.XPathExpressionException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import de.consist.bmu.rule.BMUMessageType;
import de.consist.bmu.rule.BMUMessageTypeEnum;
import de.consist.bmu.rule.BMUVersion;
import de.consist.bmu.rule.RuleFactory;
import de.consist.bmu.rule.SignatureVerificationResult;
import de.consist.bmu.rule.config.schema.fxs.FXSConfig;
import de.consist.bmu.rule.config.schema.fxs.FXSConfigHelper;
import de.consist.bmu.rule.config.schema.fxs.FXSSCHEMATYPE;
import de.consist.bmu.rule.config.schema.fxs.FXSTYPE;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.impl.FXSDokument;
import de.consist.bmu.rule.impl.SignatureVerificationResultImpl;
import de.consist.bmu.rule.util.CertUtils;
import de.consist.bmu.rule.util.DateUtils;
import de.consist.bmu.rule.util.XmlUtils;
import de.consist.bmu.rule.xpath.XPathFassade;

/**
 * Enumeration der verschiedenen Dokumenttypen.
 * 
 * @author srossbroich
 */
@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name = "messagetype", propOrder = { "_bmuMsgTypeEnum", "_isMessage", "_version", "_msgUUID", "_docNr",
		"_hasSubDocs", "_sigIds", "_signTime", "_en", "_signatureVerificationResultList", "_fxsDokumentList" })
@XmlRootElement(name = "MessageType")
public class BMUMessageTypeImpl implements BMUMessageType, Serializable {

	private static final long serialVersionUID = 1L;

	private static final String ENSN_VORLAGELAYER_NAME = "ENSNVorlageLayer";
	private static final String BGS_VORLAGELAYER_NAME = "BGSVorlageLayer";
	private static final String UNS_VORLAGELAYER_NAME = "UNSVorlageLayer";

	private static final Log LOGGER = LogFactory.getLog(BMUMessageTypeImpl.class);
	@XmlElement(name = "MessageTypeEnum", type = BMUMessageTypeEnum.class)
	private BMUMessageTypeEnum _bmuMsgTypeEnum;
	@XmlElement(name = "Nachricht")
	private boolean _isMessage;
	@XmlElement(name = "Version")
	private BMUVersion _version;
	@XmlElement(name = "MsgUUID")
	private String _msgUUID;
	@XmlElement(name = "DocNr")
	private String _docNr;
	@XmlElement(name = "SubDocs")
	private boolean _hasSubDocs;
	@XmlElement(name = "SignaturID")
	private List<String> _sigIds;
	@XmlElement(name = "SigningTime")
	private Date _signTime;
	@XmlElement(name = "EN")
	private ENStatus _en;
	@XmlElement(name = "SignatureVerificationResult")
	private List<SignatureVerificationResult> _signatureVerificationResultList;
	@XmlElement(name = "FXSDokument")
	private List<FXSDokument> _fxsDokumentList;

	/**
	 * Default Konstruktor fuer JAXB.
	 */
	public BMUMessageTypeImpl() {
	}

	/**
	 * Adapter fuer JAXB.
	 */
	public static class Adapter extends XmlAdapter<BMUMessageTypeImpl, BMUMessageType> {
		/**
		 * {@inheritDoc}
		 */
		@Override
		public final BMUMessageType unmarshal(BMUMessageTypeImpl msgTypeImpl) {
			return msgTypeImpl;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public final BMUMessageTypeImpl marshal(BMUMessageType msgType) {
			return (BMUMessageTypeImpl) msgType;
		}
	}

	/**
	 * @param bmuMsgTypeEnum
	 *            Der EnumerationsTyp
	 */
	public BMUMessageTypeImpl(BMUMessageTypeEnum bmuMsgTypeEnum) {
		_bmuMsgTypeEnum = bmuMsgTypeEnum;
		_isMessage = true;
		_version = BMUVersion.V104;
		_msgUUID = null;
		_docNr = null;
		_hasSubDocs = false;
		_sigIds = new ArrayList<String>();
		_signTime = new Date();
		_en = ENStatus.Unknown;
		_signatureVerificationResultList = new ArrayList<SignatureVerificationResult>();
		_fxsDokumentList = new ArrayList<FXSDokument>();
	}

	/**
	 * aktuelle BMU-Version (1.04).
	 */
	public static final BMUVersion BMU_SPEC_VERSION_CURRENT = BMUVersion.V104;

	private static final String BMU_SPEC_VERSION_XPATH = "/msg:Nachricht/@lib:Spezifikationsversion";
	private static final String BMU_SPEC_VERSION_NO_MSG_XPATH = "/*/@lib:Spezifikationsversion";
	private static final String XPATH_BMU_MSG = "count(/msg:Nachricht) > 0";
	private static final String XPATH_MSG_TYPE_NAME = "local-name(/msg:Nachricht/msg:Nutzdaten/*)";
	private static final String XPATH_MSG_TYPE_NAME_FXS = "descendant::*[local-name()='FreieXMLStruktur']";
	private static final String XPATH_MSG_TYPE_NAME_FXS_G11 = "descendant::*[local-name()='AnyXml' and namespace-uri()='Waste-WG-2018-Message']";
	// private static final String XPATH_MSG_TYPE_NAME_FXS_ABFAEV =
	// "local-name(descendant::msg:FreieXMLStruktur/*[namespace-uri()='" +
	// Namespace.abfaev.getUri() + "'])";
	// private static final String XPATH_MSG_TYPE_NAME_FXS_ERZST =
	// "local-name(descendant::msg:FreieXMLStruktur/*[namespace-uri()='" +
	// Namespace.erzst.getUri() + "'])";
	private static final String XPATH_NUTZDATEN_MSG_TYPE_NAME = "local-name(/*)";
	private static final String XPATH_MSGUUID = "/msg:Nachricht/@msg:MsgUUID";

	/**
	 * {@inheritDoc}
	 */
	public final String toString() {
		return _bmuMsgTypeEnum.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	public final BMUMessageTypeEnum getEnumType() {
		return _bmuMsgTypeEnum;
	}

	/**
	 * {@inheritDoc}
	 */
	public final Namespace getNS() {
		return _bmuMsgTypeEnum.getNS();
	}

	/**
	 * {@inheritDoc}
	 */
	public final String[] getXPathNummer() {
		return _bmuMsgTypeEnum.getXPathNummer();
	}

	/**
	 * {@inheritDoc}
	 */
	public final String getBMUType() {
		return _bmuMsgTypeEnum.getBMUType();
	}

	/**
	 * {@inheritDoc}
	 */
	public final boolean isLayerDoc() {
		return _bmuMsgTypeEnum.isLayerDoc();
	}

	/**
	 * {@inheritDoc}
	 */
	public final boolean isMessage() {
		return _isMessage;
	}

	/**
	 * {@inheritDoc}
	 */
	public final BMUVersion getVersion() {
		return _version;
	}

	/**
	 * {@inheritDoc}
	 */
	public final String getMsgUUID() {
		return _msgUUID;
	}

	/**
	 * {@inheritDoc}
	 */
	public final String getName() {
		return _bmuMsgTypeEnum.getNS().getPrefix() + ":" + _bmuMsgTypeEnum.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	public final String getDocNr() {
		return _docNr;
	}

	/**
	 * {@inheritDoc}
	 */
	public final List<String> getSigIds() {
		return _sigIds;
	}

	/**
	 * {@inheritDoc}
	 */
	public final Date getFirstSignTime() {
		return _signTime;
	}

	/**
	 * {@inheritDoc}
	 */
	public final Date getLastSignTime() {
		Date retVal = new Date(0L);
		for (SignatureVerificationResult svr : _signatureVerificationResultList) {
			if (svr.getSigningTime().after(retVal)) {
				retVal = svr.getSigningTime();
			}
		}
		return retVal;
	}

	/**
	 * {@inheritDoc}
	 */
	public final ENStatus getENStatus() {
		return _en;
	}

	/**
	 * {@inheritDoc}
	 */
	public final boolean isIntern() {
		if (BMUMessageTypeEnum.RegistrierungsantragZKS.equals(_bmuMsgTypeEnum)
				|| BMUMessageTypeEnum.RegistrierungsauftragZKS.equals(_bmuMsgTypeEnum)) {
			return true;
		}
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	public final boolean hasSubDocs() {
		return _hasSubDocs;
	}

	/**
	 * {@inheritDoc}
	 */
	public final List<SignatureVerificationResult> getSignatureVerificationResultList() {
		return _signatureVerificationResultList;
	}

	/**
	 * {@inheritDoc}
	 */
	public final SignatureVerificationResult getSignatureVerificationResult(String sigID) {
		SignatureVerificationResult result = null;
		for (SignatureVerificationResult svr : _signatureVerificationResultList) {
			if (sigID.equals(svr.getSignatureID())) {
				result = svr;
				break;
			}
		}
		return result;
	}

	/**
	 * @param doc
	 *            Document
	 * @return BMUMessageTypeEnum
	 * @throws BMUException
	 *             BMUException
	 */
	public static BMUMessageTypeImpl getMessageType(Document doc) throws BMUException {
		BMUMessageTypeImpl bmt = null;
		boolean isMessage;
		try {
			isMessage = XPathFassade.getInstance().evalBool(doc.getDocumentElement(), XPATH_BMU_MSG);
			String msgTypeName = null;
			String version = null;
			String msgUUID = null;
			if (isMessage) {
				msgTypeName = XPathFassade.getInstance().evaluate(doc, XPATH_MSG_TYPE_NAME);
				version = XPathFassade.getInstance().evaluate(doc, BMU_SPEC_VERSION_XPATH);
				msgUUID = XPathFassade.getInstance().evaluate(doc, XPATH_MSGUUID);
			} else {
				LOGGER.debug("Document ist keine BMU-Nachricht, prfe auf Nutzdaten..");
				msgTypeName = XPathFassade.getInstance().evaluate(doc, XPATH_NUTZDATEN_MSG_TYPE_NAME);
				version = XPathFassade.getInstance().evaluate(doc, BMU_SPEC_VERSION_NO_MSG_XPATH);
			}
			LOGGER.debug("BMU-Nachrichtentyp: " + msgTypeName + ", version=" + version);
			BMUMessageTypeEnum bmte = BMUMessageTypeEnum.valueOf(msgTypeName);
            // mey 2021.07.06 - andere BMU-Typen, die in einer freien XML-Struktur einer Mitteilung enthalten sind?
            if (BMUMessageTypeEnum.Mitteilung.equals(bmte)) {
                String testEN = XPathFassade.getInstance().evaluate(doc, BMUMessageTypeEnum.ENSNDokument.getXPathNummer()[0]);
                if (testEN != null && !testEN.isEmpty()) {
                    bmte = BMUMessageTypeEnum.ENSNDokument;
                }
            }
			// RuleConfig.ASYS.equals(RuleFactory.getInstance().getRuleConfig()))
			// {
			// LOGGER.debug("Prfe auf AbfAEV-Daten in der freien XML-Struktur
			// der Mitteilung..");
			// msgTypeName = XPathFassade.getInstance().evaluate(doc,
			// XPATH_MSG_TYPE_NAME_FXS_ABFAEV);
			// if (msgTypeName != null && msgTypeName.length() > 0) {
			// try {
			// bmte = BMUMessageTypeEnum.valueOf(msgTypeName);
			// } catch (IllegalArgumentException e) {
			// LOGGER.error("Invalid element in AbfAEV-namespace: " +
			// msgTypeName);
			// }
			// }
			// LOGGER.debug("Prfe auf ErzStamm-Daten in der freien XML-Struktur
			// der Mitteilung..");
			// msgTypeName = XPathFassade.getInstance().evaluate(doc,
			// XPATH_MSG_TYPE_NAME_FXS_ERZST);
			// if (msgTypeName != null && msgTypeName.length() > 0) {
			// try {
			// bmte = BMUMessageTypeEnum.valueOf(msgTypeName);
			// } catch (IllegalArgumentException e) {
			// LOGGER.error("Invalid element in erzst-namespace: " +
			// msgTypeName);
			// }
			// }
			// }
			bmt = new BMUMessageTypeImpl(bmte);
			bmt._isMessage = isMessage;
			bmt._version = BMUVersion.lookupVersion(version);
			bmt._msgUUID = msgUUID;
			String docNr = null;
			String[] xPathNummer = bmte.getXPathNummer();
			for (int i = 0; i < xPathNummer.length; i++) {
				try {
					docNr = XPathFassade.getInstance().evaluate(doc.getDocumentElement(), xPathNummer[i]);
					if (docNr != null && docNr.length() > 0) {
						docNr = docNr.trim();
						LOGGER.debug("Nr ermittelt: " + docNr);
						break;
					}
				} catch (XPathExpressionException ex) {
					LOGGER.debug("Nr konnte nicht ermittelt werden: " + ex.getMessage());
				}
			}
			bmt._docNr = docNr;
			bmt._signTime = new Date();
			bmt.check4SubDocs(doc);
			bmt.checkENSN(doc);
			bmt.checkSignatures(doc);
			bmt.checkFXS(doc);
		} catch (XPathExpressionException e) {
			throw new BMUException("Fehler beim Auswerten des XPath-Ausdrucks", e);
		}
		return bmt;
	}

	private void checkFXS(Document doc) throws BMUException {
		try {
			NodeList nlFXS = XPathFassade.getInstance().evaluateNodeList(doc, XPATH_MSG_TYPE_NAME_FXS);
			if (nlFXS.getLength() == 0) {
				nlFXS = XPathFassade.getInstance().evaluateNodeList(doc, XPATH_MSG_TYPE_NAME_FXS_G11);
			}
			FXSConfig fxsConfig = RuleFactory.getInstance().getFXSConfig();
			if (fxsConfig != null) {
				LOGGER.debug("Pruefe auf konfigurierte freie XML-Strukturen im Dokument");
				for (int i = 0; i < nlFXS.getLength(); i++) {
					Element elemFXS = (Element) nlFXS.item(i);
					// FXSTYPE fxsType = null;
					Element elemData = XmlUtils.getFirstChildElement(elemFXS);
					String rootElementName = elemData.getLocalName();
					String nsURI = elemData.getNamespaceURI();
					LOGGER.debug("Pruefe auf Element: " + nsURI + ":" + rootElementName);
					String fxsName = null;
					for (FXSTYPE fxs : fxsConfig.getFXS()) {
						FXSSCHEMATYPE fxsSchema = FXSConfigHelper.getSchemaByID(fxsConfig, fxs.getSchemaID());
						if (fxsSchema != null && fxsSchema.getNSUri().equals(nsURI)
								&& fxs.getRootElementName().contains(rootElementName)) {
							if (fxsName == null) {
								fxsName = fxs.getName();
							}
							if (fxsName.equals(fxs.getName())) {
								String id = elemFXS.getAttributeNS(Namespace.TypenBibliothek.getUri(), "id");
								String uri = elemFXS.getAttributeNS(Namespace.TypenBibliothek.getUri(), "NamespaceURI");
								FXSDokument fxsDok = new FXSDokument(fxs, id, uri, elemData);
								LOGGER.debug("Freie XML-Struktur gefunden: " + fxsDok);
								_fxsDokumentList.add(fxsDok);
							} else {
								LOGGER.warn("Konfigurierte freie Xml-Strukturen unterschiedlicher Typen gefunden: " + fxsName + ", " + fxs.getName());
							}
						}
					}
				}
			} else {
				LOGGER.debug("Keine Konfiguration von freien XML-Strukturen vorhanden");
			}
		} catch (XPathExpressionException e) {
			LOGGER.error("Fehler beim Prfen auf bekannte freie XML-Strukturen.", e);
			throw new BMUException("Fehler beim Prfen auf bekannte freie XML-Strukturen.", e);
		}
	}

	private void check4SubDocs(Document doc) throws BMUException {
		if (BMUMessageTypeEnum.ENSNDokument.equals(_bmuMsgTypeEnum)) {
			try {
				_hasSubDocs = XPathFassade.getInstance().evalBool(doc.getDocumentElement(),
						"count(descendant::en:EGFDokument) > 0")
						|| XPathFassade.getInstance().evalBool(doc.getDocumentElement(),
								"count(descendant::ags:AGSBescheid) > 0")
						|| XPathFassade.getInstance().evalBool(doc.getDocumentElement(),
								"count(descendant::en:DADokument) > 0");
				LOGGER.debug("Integrierte Dokumente gefunden: " + _hasSubDocs);
			} catch (XPathExpressionException e) {
				LOGGER.error("Fehler beim Prfen auf integrierte Dokumente.", e);
				throw new BMUException("Fehler beim Prfen auf integrierte Dokumente.", e);
			}
		}
	}

	/**
	 * Ermittelt den Signaturzeitpunkt. Signaturen im Vorlagelayer von ENS, BGS
	 * und UNS werden ignoriert.
	 * 
	 * @param doc
	 */
	private void checkSignatures(Document doc) throws BMUException {
		_signatureVerificationResultList.clear();
		_sigIds.clear();
		try {
			XPathFassade xf = XPathFassade.getInstance();
			NodeList nl = xf.evaluateNodeList(doc, "/descendant::ds:Signature");
			XMLGregorianCalendar signingCal = null;
			for (int i = 0; i < nl.getLength(); i++) {
				Element sigNode = (Element) nl.item(i);
				String sigID = sigNode.getAttribute("Id");
				if (sigID == null || sigID.isEmpty()) {
				    LOGGER.warn("Die Signatur im Element '" + xf.evaluate(sigNode, "local-name(parent::*)") + "' hat kein ID-Attribut!");
				    continue;
				}
				_sigIds.add(sigID);
				String subjectDN = null;
				NodeList certNodeList = sigNode.getElementsByTagNameNS(Namespace.xmldsig.getUri(), "X509Certificate");
				if (certNodeList.getLength() == 0) {
					LOGGER.warn("Kein Zertifikat in der Signatur mit der ID '" + sigID + "' vorhanden");
				} else if (certNodeList.getLength() > 1) {
					LOGGER.warn("Mehr als 1 Zertifikat in der Signatur mit der ID '" + sigID + "' vorhanden");
				} else {
					Element elemCert = (Element) certNodeList.item(0);
					X509Certificate x509Cert = CertUtils.getX509CertificateFromBase64(elemCert.getTextContent());
					// subjectDN = CertUtils.getSubjectCN(x509Cert);
					subjectDN = x509Cert.getSubjectX500Principal().getName();
				}
				boolean vorlageLayer = false;
				String sigNodeParentLocalName = xf.evaluate(sigNode, "local-name(parent::*)");
				if (ENSN_VORLAGELAYER_NAME.equals(sigNodeParentLocalName)
						|| BGS_VORLAGELAYER_NAME.equals(sigNodeParentLocalName)
						|| UNS_VORLAGELAYER_NAME.equals(sigNodeParentLocalName)) {
					LOGGER.debug("skipping SigningTime for Signature in Layer: " + sigNodeParentLocalName);
					vorlageLayer = true;
				}
				Date signingTime = null;
				NodeList signingTimeNodeList = sigNode.getElementsByTagNameNS(Namespace.xades.getUri(), "SigningTime");
				if (signingTimeNodeList.getLength() > 0) {
					Node signingTimeNode = signingTimeNodeList.item(0);
					XMLGregorianCalendar actSigningCal = DatatypeFactory.newInstance()
							.newXMLGregorianCalendar(signingTimeNode.getTextContent());
					signingTime = actSigningCal.toGregorianCalendar().getTime();
					if (!vorlageLayer) {
						if (signingCal == null) {
							signingCal = actSigningCal;
						} else {
							if (actSigningCal.compare(signingCal) == DatatypeConstants.LESSER) {
								signingCal = actSigningCal;
							}
						}
					}
				}
                boolean eIDAS = false;
                NodeList nlCertV2 = sigNode.getElementsByTagNameNS(Namespace.xades.getUri(), "SigningCertificateV2");
                if (nlCertV2.getLength() > 0) {
                    LOGGER.debug("Element 'SigningCertificateV2' vorhanden!");
                    eIDAS = true;
                }
				_signatureVerificationResultList
                        .add(new SignatureVerificationResultImpl(sigID, subjectDN, signingTime, eIDAS));
			}
			if (signingCal != null) {
				_signTime = signingCal.toGregorianCalendar().getTime();
				LOGGER.debug("SigningTime: " + DateUtils.toDateTimeString(_signTime));
			}
		} catch (XPathExpressionException e) {
			LOGGER.error("Fehler beim Verarbeiten der Signaturen.", e);
			throw new BMUException("Fehler beim Verarbeiten der Signaturen.", e);
		} catch (DOMException e) {
			LOGGER.error("Fehler beim Ermitteln des Signaturzeitpunkts.", e);
			throw new BMUException("Fehler beim Ermitteln des Signaturzeitpunkts.", e);
		} catch (DatatypeConfigurationException e) {
			LOGGER.error("Fehler beim Ermitteln des Signaturzeitpunkts.", e);
			throw new BMUException("Fehler beim Ermitteln des Signaturzeitpunkts.", e);
		} catch (CertificateException e) {
			LOGGER.error("Fehler beim Verarbeiten der Signatur.", e);
			throw new BMUException("Fehler beim Ermitteln des Signaturzeitpunkts.", e);
		} catch (NoSuchProviderException e) {
			LOGGER.error("Fehler beim Verarbeiten der Signatur.", e);
			throw new BMUException("Fehler beim Ermitteln des Signaturzeitpunkts.", e);
		}
	}

	/**
	 * @param doc
	 */
	private void checkENSN(Document doc) throws BMUException {
		if (BMUMessageTypeEnum.ENSNDokument.equals(_bmuMsgTypeEnum)) {
			try {
				XPathFassade xpf = XPathFassade.getInstance();
				Element docElem = doc.getDocumentElement();
				boolean indicatorENSet = xpf.evalBool(docElem, "count(/descendant::en:IndicatorEN) > 0");
				if (indicatorENSet) {
					boolean indicatorEN = xpf.evalBool(docElem,
							"/descendant::en:IndicatorEN[last()]='true' or /descendant::en:IndicatorEN[last()]=1");
					if (indicatorEN) {
						_en = ENStatus.EN;
					} else {
						_en = ENStatus.SN;
					}
				} else {
					LOGGER.error("IndicatorEN ist nicht gesetzt, Klassifikation in EN/SN nicht mglich!");
				}
			} catch (XPathExpressionException e) {
				LOGGER.error("Fehler bei der Klassifizierung auf EN/SN", e);
				throw new BMUException("Fehler bei der Klassifizierung auf EN/SN", e);
			}
		}
	}

	@Override
	public List<FXSDokument> getFXSDokumentList() {
		return _fxsDokumentList;
	}
}
