/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2024-02-06 14:29:28 +0000 (Tue, 06 Feb 2024) $ $Rev: 1787 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.def;

import jakarta.xml.bind.annotation.XmlAccessType;
import jakarta.xml.bind.annotation.XmlAccessorType;
import jakarta.xml.bind.annotation.XmlElement;
import jakarta.xml.bind.annotation.XmlRootElement;
import jakarta.xml.bind.annotation.XmlType;
import jakarta.xml.bind.annotation.adapters.XmlAdapter;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import de.consist.bmu.rule.MeldungTyp;
import de.consist.bmu.rule.schema.Namespace;

/**
 * Klasse fr die Meldung in der Regeldefinition.
 * Wird mit jaxb als xml serialisiert.
 */
@XmlAccessorType(XmlAccessType.FIELD)
@XmlType(name = "meldungtyp", propOrder = { /* "_version",*/ "_klasse", "_stufe",
        "_beschreibung", "_code", "_abhilfe" })
@XmlRootElement(name = "Meldung")
public class MeldungTypImpl implements MeldungTyp {

    private static final int INT_6 = 6;

    private static final int INT_10 = 10;

    private static final int INT_100 = 100;

    private static final int INT_1000 = 1000;

    private static final int INT_252 = 252;

    private static final int INT_255 = 255;

    private static final Log LOGGER = LogFactory.getLog(MeldungTypImpl.class);
    
    /** */
    private static final long serialVersionUID = 1L;

//    @XmlElement(name = "Version")
//    @XmlJavaTypeAdapter(XmlBMUVersionAdapter.class)
//    private BMUVersion _version;
    @XmlElement(name = "Klasse")
    private FehlerKlasse _klasse;
    @XmlElement(name = "Stufe")
    private FehlerStufe _stufe;
    @XmlElement(name = "Beschreibung")
    private String _beschreibung;
    @XmlElement(name = "Code")
    private String _code;
    @XmlElement(name = "Abhilfe")
    private String _abhilfe;

    /**
     * Konstruktor. 
     */
    public MeldungTypImpl() {
    }

    /**
     * Adapter fuer JAXB.
     */
    public static class Adapter extends XmlAdapter<MeldungTypImpl, MeldungTyp> {
        /**
         * {@inheritDoc}
         */
        @Override
        public final MeldungTyp unmarshal(MeldungTypImpl meldungTypImpl) {
            return meldungTypImpl;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public final MeldungTypImpl marshal(MeldungTyp meldungTyp) {
            return (MeldungTypImpl) meldungTyp;
        }
    }

    /**
     * @param klasse Die Fehlerklasse
     * @param stufe Die Fehlerstufe
     * @param beschreibung Die Fehlerbeschreibung
     * @param code Der Fehlercode
     * @param abhilfe Der Abhilfetext
     */
    public MeldungTypImpl(FehlerKlasse klasse, FehlerStufe stufe,
            String beschreibung, String code, String abhilfe) {
        this._klasse = klasse;
        this._stufe = stufe;
        this._beschreibung = beschreibung;
        this._code = code;
        this._abhilfe = abhilfe;
//        this._version = BMUVersion.V104;
    }

    /**
     * Copy-Konstruktor.
     * @param source MeldungTyp
     */
    public MeldungTypImpl(MeldungTyp source) {
        this._klasse = source.getKlasse();
        this._stufe = source.getStufe();
        this._beschreibung = source.getBeschreibung();
        this._code = source.getCode();
        this._abhilfe = source.getAbhilfe();
    }

    /**
     * {@inheritDoc}
     */
    public final FehlerKlasse getKlasse() {
        return _klasse;
    }

    /**
     * {@inheritDoc}
     */
    public final void setKlasse(FehlerKlasse klasse) {
        this._klasse = klasse;
    }

    /**
     * {@inheritDoc}
     */
    public final FehlerStufe getStufe() {
        return _stufe;
    }

    /**
     * {@inheritDoc}
     */
    public final void setStufe(FehlerStufe stufe) {
        this._stufe = stufe;
    }

    /**
     * {@inheritDoc}
     */
    public final String getBeschreibung() {
        return _beschreibung;
    }

    /**
     * {@inheritDoc}
     */
    public final void setBeschreibung(String beschreibung) {
        this._beschreibung = beschreibung;
    }

    /**
     * {@inheritDoc}
     */
    public final String getCode() {
        return _code;
    }

    /**
     * {@inheritDoc}
     */
    public final void setCode(String code) {
        this._code = code;
    }

    /**
     * {@inheritDoc}
     */
    public final String getAbhilfe() {
        return _abhilfe;
    }

    /**
     * {@inheritDoc}
     */
    public final void setAbhilfe(String abhilfe) {
        this._abhilfe = abhilfe;
    }

//    /* (non-Javadoc)
//     * @see de.consist.bmu.rule.def.MeldungTyp#getVersion()
//     */
//    @Override
//    public final BMUVersion getVersion() {
//        return _version;
//    }

    /**
     * Ereugt ein Xml-Element aus der Meldung.
     * 
     * @param doc Document
     * @param message String
     * @param index int
     * @return Element
     */
    public final Element toXml(Document doc, String message, int index) {
        Element meldung = doc.createElementNS(Namespace.Nachricht.getUri(), Namespace.Nachricht.getPrefix() + ":Meldung");
        Element klasse = doc.createElementNS(Namespace.Nachricht.getUri(), Namespace.Nachricht.getPrefix() + ":Klasse");
        klasse.setTextContent(getKlasse().toString());
        meldung.appendChild(klasse);
        Element stufe = doc.createElementNS(Namespace.Nachricht.getUri(), Namespace.Nachricht.getPrefix() + ":Stufe");
        stufe.setTextContent(getStufe().toString());
        meldung.appendChild(stufe);
        Element beschreibung = doc.createElementNS(Namespace.Nachricht.getUri(), Namespace.Nachricht.getPrefix() + ":Beschreibung");
        String tempBeschreibung = getBeschreibung();
        if (message != null) {
            tempBeschreibung += message;
        }
        if (tempBeschreibung.length() > INT_255) {
            tempBeschreibung = tempBeschreibung.substring(0, INT_252);
            tempBeschreibung += "...";
            LOGGER.warn("Beschreibung ist lnger als 255 Zeichen, Inhalt wird abgeschnitten: " + tempBeschreibung);
        }
        beschreibung.setTextContent(tempBeschreibung);
        meldung.appendChild(beschreibung);
        String tempCode = getCode();
        if (tempCode != null && index >= 0 && index < INT_1000) {
            Element code = doc.createElementNS(Namespace.Nachricht.getUri(), Namespace.Nachricht.getPrefix() + ":Code");
            String indexString = null;
            if (index < INT_10) {
                indexString = "00" + index;
            } else if (index < INT_100) {
                indexString = "0" + index;
            } else {
                indexString = Integer.toString(index);
            }
            if (tempCode.length() > INT_6) {
                tempCode = tempCode.substring(0, INT_6);
                LOGGER.warn("Code wurde abgeschnitten: " + tempCode);
            }
            tempCode = tempCode + "_" + indexString;
            code.setTextContent(tempCode);
            meldung.appendChild(code);
        }
        if (getAbhilfe() != null) {
            Element abhilfe = doc.createElementNS(Namespace.Nachricht.getUri(), Namespace.Nachricht.getPrefix() + ":Abhilfe");
            abhilfe.setTextContent(getAbhilfe());
            meldung.appendChild(abhilfe);
        }
        return meldung;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public final String toString() {
        StringBuilder builder = new StringBuilder();
//        builder.append("Version=" + _version);
        builder.append("Klasse=" + _klasse);
        builder.append(", Stufe=" + _stufe);
        builder.append(", Beschreibung=" + _beschreibung);
        builder.append(", Code=" + _code);
        builder.append(", Abhilfe=" + _abhilfe);
        return builder.toString();
    }
}
