/**
q * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2021-09-07 20:13:18 +0000 (Tue, 07 Sep 2021) $ $Rev: 1662 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */
package de.consist.bmu.rule.impl;

import java.util.ArrayList;
import java.util.List;

import javax.xml.xpath.XPathExpressionException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXParseException;

import de.consist.bmu.rule.BMUDokument;
import de.consist.bmu.rule.BMUMessageTypeEnum;
import de.consist.bmu.rule.RuleDef;
import de.consist.bmu.rule.RuleResult;
import de.consist.bmu.rule.error.BMUException;
import de.consist.bmu.rule.schema.SchemaValidator;
import de.consist.bmu.rule.schema.SchemaValidator.ValidationErrorHandler;
import de.consist.bmu.rule.util.XmlUtils;
import de.consist.bmu.rule.xpath.XPathFassade;

/**
 * Prft, ob an den spezifizierten Positionen ein Element mit dem erwarteten
 * Namen und dem entsprechenden Namespace-URI existiert. Die Substruktur wird
 * gegen das entsprechende Schema geprft.
 * 
 * @author srossbroich
 */
public final class RuleImplFXSTyp extends RuleImpl {

    private static final long serialVersionUID = 1L;

    private static enum FXSDoc {
        AGSBESCHEID("AGSBescheid", "uri:BMU_Waste_Interface/AGS"), 
        DA("Deklarationsanalyse", "uri:BMU_Waste_Interface/DA"), 
        NTZ("EUDINNotificationDocument", "usr:EUDINNotificationDocument:2:1"), 
        VBF_WR("EUDINCertificateOfWasteReceiptDocument", "usr:EUDINCertificateOfWasteReceiptDocument:2:1"), 
        VBF_WD("EUDINCertificateOfWasteRecoveryDisposalDocument", "usr:EUDINCertificateOfWasteRecoveryDisposalDocument:2:1"), 
        VBF_WM("EUDINWasteMovementDocument", "usr:EUDINWasteMovementDocument:2:1"), 
        COMR("EUDINConfirmationOfMessageReceipt", "usr:EUDINConfirmationOfMessageReceipt:2:1"), 
        WTS("EUDINWasteTransportStatement", "usr:EUDINWasteTransportStatement:2:1"), 
        BTR_UM("Antragsobjekt", "urn:de:bmu:eanv:zks:ZKSBetriebUmhang:1:04"),
        NTZ_G11("Message", "Waste-WG-2018-Message"),
        VBF_G11("Message", "Waste-WG-2018-Message"); 
//        Anzeige("Anzeige", Namespace.abfaev.getUri()), 
//        AnzeigeUnterlagen("AnzeigeUnterlagen", Namespace.abfaev.getUri()), 
//        ErlaubnisAntrag("ErlaubnisAntrag", Namespace.abfaev.getUri()), 
//        ErlaubnisUnterlagen("ErlaubnisUnterlagen", Namespace.abfaev.getUri()), 
//        AsysRueckantwort("AsysRueckantwort", Namespace.abfaev.getUri()), 
//        Erzeuger("Erzeuger", Namespace.erzst.getUri());

        private String _nodeName;
        private String _nsURI;

        private FXSDoc(String nodeName, String nsURI) {
            _nodeName = nodeName;
            _nsURI = nsURI;
        }
    }

    private static enum FXSTyp {
        AGSBESCHEID("/descendant::en:AGSBescheid/*",
                new FXSDoc[] { FXSDoc.AGSBESCHEID },
                new BMUMessageTypeEnum[] { BMUMessageTypeEnum.ENSNDokument }), 
        DA("/descendant::en:Deklarationsanalyse/*",
                new FXSDoc[] { FXSDoc.DA }, new BMUMessageTypeEnum[] {
                        BMUMessageTypeEnum.ENSNDokument,
                        BMUMessageTypeEnum.DADokument,
                        BMUMessageTypeEnum.RegisterAuszug }), 
        NTZ("/descendant::ntz:Notifzierung/*",
                new FXSDoc[] { FXSDoc.NTZ, FXSDoc.NTZ_G11 },
                new BMUMessageTypeEnum[] { BMUMessageTypeEnum.Abfallverbringungsdokument }), 
        VBF("/descendant::ntz:VersandBegleitformular/*",
                new FXSDoc[] { FXSDoc.VBF_WR, FXSDoc.VBF_WD, FXSDoc.VBF_WM, FXSDoc.VBF_G11 },
                new BMUMessageTypeEnum[] { BMUMessageTypeEnum.Abfallverbringungsdokument }), 
        COMR("/descendant::ntz:CoMR-Dokument/*",
                new FXSDoc[] { FXSDoc.COMR },
                new BMUMessageTypeEnum[] { BMUMessageTypeEnum.Abfallverbringungsdokument }), 
        WTS("/descendant::ntz:WasteTransportStatement/*",
                new FXSDoc[] { FXSDoc.WTS },
                new BMUMessageTypeEnum[] { BMUMessageTypeEnum.Abfallverbringungsdokument }), 
        BTR_UM("/descendant::zks:FreieXMLStruktur[parent::zks:Antrag]/*",
                new FXSDoc[] { FXSDoc.BTR_UM }, new BMUMessageTypeEnum[] {
                        BMUMessageTypeEnum.Registrierungsantrag,
                        BMUMessageTypeEnum.RegistrierungsantragZKS }); 
//        Anzeige("/descendant::msg:FreieXMLStruktur/*",
//                new FXSDoc[] { FXSDoc.Anzeige },
//                new BMUMessageTypeEnum[] { BMUMessageTypeEnum.Anzeige }), 
//        AnzeigeUnterlagen("/descendant::msg:FreieXMLStruktur/*",
//                new FXSDoc[] { FXSDoc.AnzeigeUnterlagen },
//                new BMUMessageTypeEnum[] { BMUMessageTypeEnum.AnzeigeUnterlagen }), 
//        ErlaubnisAntrag("/descendant::msg:FreieXMLStruktur/*",
//                new FXSDoc[] { FXSDoc.ErlaubnisAntrag },
//                new BMUMessageTypeEnum[] { BMUMessageTypeEnum.ErlaubnisAntrag }), 
//        ErlaubnisUnterlagen("/descendant::msg:FreieXMLStruktur/*",
//                new FXSDoc[] { FXSDoc.ErlaubnisUnterlagen },
//                new BMUMessageTypeEnum[] { BMUMessageTypeEnum.ErlaubnisUnterlagen }), 
//        AsysRueckantwort("/descendant::msg:FreieXMLStruktur/*",
//                new FXSDoc[] { FXSDoc.AsysRueckantwort },
//                new BMUMessageTypeEnum[] { BMUMessageTypeEnum.AsysRueckantwort }), 
//        Erzeuger("/descendant::msg:FreieXMLStruktur/*",
//                new FXSDoc[] { FXSDoc.Erzeuger },
//                new BMUMessageTypeEnum[] { BMUMessageTypeEnum.Erzeuger });
        private String _xPath;
        private FXSDoc[] _fxsDocs;
        private BMUMessageTypeEnum[] _parentTypes;

        private FXSTyp(String xPath, FXSDoc[] fxsDocs,
                BMUMessageTypeEnum[] parentTypes) {
            _xPath = xPath;
            _fxsDocs = fxsDocs;
            _parentTypes = parentTypes;
        }
    }

    private static final Log LOGGER = LogFactory.getLog(RuleImplFXSTyp.class);

    /**
     * @param ruleDef
     *            RuleDef
     */
    public RuleImplFXSTyp(RuleDef ruleDef) {
        super(ruleDef);
    }

    private List<RuleResult> validateSubDoc(Node node, int index, String id)
            throws BMUException {
        LOGGER.debug("Validating sub-structure with id: " + id);
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        Document doc = XmlUtils.newDocument(true);
        Node importedNode = doc.importNode(node, true);
        doc.appendChild(importedNode);
        // Wegen xsi:type Attribut in ECDSA-Signatur..
        doc.normalizeDocument();
        ValidationErrorHandler handler = SchemaValidator.getInstance()
                .validate(doc.getDocumentElement());
        if (handler != null) {
            List<SAXParseException> errorList = handler.getErrorList();
            if (!errorList.isEmpty()) {
                for (SAXParseException saxParseException : errorList) {
                    ruleResultList
                            .add(new RuleResultImpl(this.getRuleDef(), index++,
                                    id + ", " + saxParseException.getMessage()));
                }
            }
        } else {
            LOGGER.error("Error validating document");
            throw new BMUException("Error validating document");
        }
        return ruleResultList;
    }

    /**
     * {@inheritDoc}
     */
    public List<RuleResult> execute(BMUDokument bmuDok) throws BMUException {
        List<RuleResult> ruleResultList = new ArrayList<RuleResult>();
        Document doc = bmuDok.getDocument();
        NodeList nodeList = null;
        Element elem = null;
        int index = 1;
        for (FXSTyp fxsTyp : FXSTyp.values()) {
            boolean typeMatch = false;
            for (BMUMessageTypeEnum msgType : fxsTyp._parentTypes) {
                if (msgType.equals(bmuDok.getMessageType().getEnumType())) {
                    typeMatch = true;
                    if (LOGGER.isTraceEnabled()) {
                        LOGGER.trace("type match: " + msgType);
                    }
                }
            }
            if (typeMatch) {
                try {
                    nodeList = XPathFassade.getInstance().evaluateNodeList(doc,
                            fxsTyp._xPath);
                    for (int i = 0; i < nodeList.getLength(); i++) {
                        elem = (Element) nodeList.item(i);
                        String id = XPathFassade.getInstance().evaluate(elem,
                                "parent::*/@lib:id");
                        boolean docMatch = false;
                        for (FXSDoc fxsDoc : fxsTyp._fxsDocs) {
                            if (fxsDoc._nodeName.equals(elem.getLocalName())
                                    && fxsDoc._nsURI.equals(elem
                                            .getNamespaceURI())) {
                                docMatch = true;
                                if (LOGGER.isTraceEnabled()) {
                                    LOGGER.trace("doc match: " + fxsDoc);
                                }
                                break;
                            }
                        }
                        if (!docMatch) {
                            ruleResultList.add(new RuleResultImpl(this
                                    .getRuleDef(), index++, id));
                        } else {
                            List<RuleResult> rrl = validateSubDoc(elem, index,
                                    id);
                            index += rrl.size();
                            ruleResultList.addAll(rrl);
                        }
                    }
                } catch (XPathExpressionException e) {
                    throw new BMUException("Fehler beim Prfen", e);
                }
            }
        }
        return ruleResultList;
    }

    /**
     * @return Die technische Dokumentation der Implementierung
     */
    public static String getTechDoc() {
        StringBuilder builder = new StringBuilder();
        builder.append("XPath fr AGSBescheid: ");
        builder.append(FXSTyp.AGSBESCHEID._xPath);
        builder.append(RuleImpl.NL);
        builder.append("XPath fr Deklarationsanalyse: ");
        builder.append(FXSTyp.DA._xPath);
        builder.append(RuleImpl.NL);
        builder.append("XPath fr Notifizierung: ");
        builder.append(FXSTyp.NTZ._xPath);
        builder.append(RuleImpl.NL);
        builder.append("XPath fr Versandbegleitformular: ");
        builder.append(FXSTyp.VBF);
        builder.append(RuleImpl.NL);
        builder.append("XPath fr EUDINConfirmationOfMessageReceipt: ");
        builder.append(FXSTyp.COMR);
        builder.append(RuleImpl.NL);
        builder.append("XPath fr EUDINWasteTransportStatement: ");
        builder.append(FXSTyp.WTS);
        builder.append(RuleImpl.NL);
        builder.append("XPath fr FreieXMLStruktur in Registrierungsantrag: ");
        builder.append(FXSTyp.BTR_UM);
        return builder.toString();
    }
}
