/**
 * BMUPruefBibliothek
 * $Author: srossbroich $ $Date: 2012-09-14 12:34:45 +0000 (Fri, 14 Sep 2012) $ $Rev: 926 $
 * Copyright 2012 by Consist ITU Environmental Software GmbH
 */

package de.consist.bmu.rule.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import de.consist.bmu.rule.error.BMUException;

/**
 * Utility Klasse fuer Byte-Arrays. Hier gefunden:
 * 
 * @see <a href="http://www.tek-tips.com/viewthread.cfm?qid=1275859">Java -
 *      Search Byte Array for Pattern</a>
 * @author jannighoefer
 */
public final class ByteUtils {

    private static final int BUFFERSIZE_DEFAULT = 1024;
    private static final Log LOGGER = LogFactory.getLog(ByteUtils.class);

    private ByteUtils() {
    }

    /**
     * Liest ein komplettes Bytearray aus einem Stream und schliesst ihn.
     * 
     * @param is
     *            InputStream
     * @return byte[]
     * @throws BMUException
     *             BMUException
     */
    public static byte[] readFromStream(InputStream is) throws BMUException {
        return readFromStream(is, BUFFERSIZE_DEFAULT);
    }

    /**
     * Liest ein komplettes Bytearray aus einem Stream und schliesst ihn.
     * 
     * @param is
     *            InputStream
     * @param buffersize
     *            int
     * @return byte[]
     * @throws BMUException
     *             BMUException
     */
    public static byte[] readFromStream(InputStream is, int buffersize)
            throws BMUException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream(buffersize);
        byte[] data = new byte[buffersize];
        int len = 0;
        try {
            while ((len = is.read(data)) != -1) {
                bos.write(data, 0, len);
            }
            is.close();
        } catch (IOException e) {
            LOGGER.error("error reading from stream", e);
            throw new BMUException("error reading from stream", e);
        }
        return bos.toByteArray();
    }

    /**
     * @param name
     *            String
     * @param buffersize
     *            int
     * @return byte[]
     * @throws BMUException
     *             BMUException
     */
    public static byte[] readFromResource(String name, int buffersize)
            throws BMUException {
        InputStream is = XmlUtils.class.getResourceAsStream(name);
        if (is == null) {
            LOGGER.error("error loading resource: " + name);
            throw new BMUException("error loading resource: " + name);
        }
        return readFromStream(is, buffersize);
    }

    /**
     * @param name
     *            String
     * @return byte[]
     * @throws BMUException
     *             BMUException
     */
    public static byte[] readFromResource(String name) throws BMUException {
        InputStream is = XmlUtils.class.getResourceAsStream(name);
        if (is == null) {
            LOGGER.error("error loading resource: " + name);
            throw new BMUException("error loading resource: " + name);
        }
        return readFromStream(is);
    }

    /**
     * @param filename
     *            String
     * @return byte[]
     * @throws BMUException
     *             BMUException
     */
    public static byte[] readFromFile(String filename) throws BMUException {
        byte[] data = null;
        File f = new File(filename);
        if (f.exists()) {
            try {
                data = readFromStream(new FileInputStream(f));
            } catch (Exception ex) {
                LOGGER.error("error reading Document from file: "
                        + f.getAbsolutePath());
                throw new BMUException("error reading Document from file: "
                        + f.getAbsolutePath(), ex);
            }
        } else {
            LOGGER.error("File not found: " + f.getAbsolutePath());
            throw new BMUException("File not found: " + f.getAbsolutePath());
        }
        return data;
    }

    /**
     * @param filename
     *            String
     * @param data
     *            byte[]
     * @throws BMUException
     *             BMUException
     */
    public static void writeToFile(String filename, byte[] data)
            throws BMUException {
        File f = new File(filename);
        try {
            ByteArrayInputStream bais = new ByteArrayInputStream(data);
            ReadableByteChannel inChannel = Channels.newChannel(bais);
            FileOutputStream fos = new FileOutputStream(f);
            FileChannel outChannel = fos.getChannel();
            outChannel.transferFrom(inChannel, 0, data.length);
            outChannel.close();
            fos.close();
        } catch (Exception e) {
            LOGGER.error("error writing to file: " + f.getAbsolutePath(), e);
            throw new BMUException("error writing to file: "
                    + f.getAbsolutePath(), e);
        }
    }

    /**
     * Sucht in einem Bytearray nach einem Bytearray.
     * 
     * @param data
     *            byte[]
     * @param pattern
     *            byte[]
     * @return boolean
     */
    public static boolean match(byte[] data, byte[] pattern) {
        int iPattern = 0;
        for (int i = 0; i < data.length; i++) {
            if (data[i] == pattern[iPattern]) {
                iPattern++;
            } else {
                if (data[i] == pattern[0]) {
                    // Entspricht dem Anfang von pattern
                    iPattern = 1; //
                } else {
                    // Am Anfang von pattern suchen
                    iPattern = 0;
                }
            }

            if (iPattern == pattern.length) {
                // Bis zum Ende von pattern gekommen
                return true;
            }
        }

        return false;
    }
}
